'use client';

import { useState, useEffect, useCallback } from 'react';
import { Event, EventFilters, EventType } from '@/types/event';
import Header from '@/components/Header';
import Hero from '@/components/Hero';
import SearchFilters from '@/components/SearchFilters';
import CategorySection from '@/components/CategorySection';
import FeaturedEvents from '@/components/FeaturedEvents';
import EventCard, { EventCardSkeleton } from '@/components/EventCard';
import Footer from '@/components/Footer';
import { motion } from 'framer-motion';
import { Calendar, ArrowDown } from 'lucide-react';

/**
 * Main page component - Light Theme with wider layout
 */
export default function Home() {
  const [events, setEvents] = useState<Event[]>([]);
  const [filteredEvents, setFilteredEvents] = useState<Event[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [countries, setCountries] = useState<string[]>([]);
  const [eventTypes, setEventTypes] = useState<EventType[]>([]);
  const [filters, setFilters] = useState<EventFilters>({
    search: '',
    eventType: 'all',
    country: 'all',
    dateRange: 'all',
    ticketAvailable: null,
  });

  // Fetch events on mount
  useEffect(() => {
    fetchEvents();
  }, []);

  // Apply filters when they change
  useEffect(() => {
    applyFilters();
  }, [filters, events]);

  const fetchEvents = async () => {
    try {
      setIsLoading(true);
      const response = await fetch('/api/events');
      const data = await response.json();
      
      setEvents(data.events);
      setFilteredEvents(data.events);
      setCountries(data.filters.countries);
      setEventTypes(data.filters.eventTypes);
    } catch (error) {
      console.error('Error fetching events:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const applyFilters = useCallback(() => {
    const params = new URLSearchParams();
    
    if (filters.search) params.set('search', filters.search);
    if (filters.eventType !== 'all') params.set('eventType', filters.eventType);
    if (filters.country !== 'all') params.set('country', filters.country);
    if (filters.dateRange !== 'all') params.set('dateRange', filters.dateRange);
    if (filters.ticketAvailable !== null) params.set('ticketAvailable', String(filters.ticketAvailable));

    fetch(`/api/events?${params.toString()}`)
      .then(res => res.json())
      .then(data => {
        setFilteredEvents(data.events);
      })
      .catch(console.error);
  }, [filters]);

  const handleCategoryClick = (category: EventType | 'all') => {
    setFilters(prev => ({ ...prev, eventType: category }));
  };

  // Get featured events (largest attendee count first)
  const featuredEvents = [...events]
    .sort((a, b) => {
      const getAttendees = (range: string) => {
        const match = range.replace(/,/g, '').match(/\d+/);
        return match ? parseInt(match[0]) : 0;
      };
      return getAttendees(b.attendeesRange) - getAttendees(a.attendeesRange);
    })
    .slice(0, 5);

  return (
    <div className="min-h-screen bg-white">
      <Header />
      
      <main>
        {/* Hero Section */}
        <Hero />

        {/* Main Content - Wider container */}
        <div className="max-w-[1400px] mx-auto px-4 sm:px-6 lg:px-8 xl:px-12 py-12">
          {/* Featured Events */}
          {!isLoading && featuredEvents.length > 0 && (
            <FeaturedEvents events={featuredEvents} />
          )}

          {/* Search & Filters */}
          <section className="py-8">
            <SearchFilters
              filters={filters}
              onFilterChange={setFilters}
              countries={countries}
              eventTypes={eventTypes}
              totalResults={filteredEvents.length}
            />
          </section>

          {/* Category Pills */}
          <CategorySection 
            onCategoryClick={handleCategoryClick}
            activeCategory={filters.eventType}
          />

          {/* Events Grid */}
          <section className="py-8">
            <div className="flex items-center justify-between mb-6">
              <div className="flex items-center gap-2">
                <Calendar className="w-5 h-5 text-[#E11D2E]" />
                <h2 className="text-xl font-heading font-semibold text-gray-900">
                  {filters.eventType === 'all' ? 'All Events' : `${filters.eventType.charAt(0).toUpperCase() + filters.eventType.slice(1).replace('-', ' ')}s`}
                </h2>
              </div>
              
              <div className="flex items-center gap-2 text-sm text-gray-500">
                <ArrowDown className="w-4 h-4" />
                <span>Sorted by date</span>
              </div>
            </div>

            {isLoading ? (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
                {[...Array(8)].map((_, i) => (
                  <EventCardSkeleton key={i} />
                ))}
              </div>
            ) : filteredEvents.length > 0 ? (
              <motion.div 
                className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6"
                initial="hidden"
                animate="visible"
                variants={{
                  visible: {
                    transition: {
                      staggerChildren: 0.05,
                    },
                  },
                }}
              >
                {filteredEvents.map((event, index) => (
                  <EventCard key={event.id} event={event} index={index} />
                ))}
              </motion.div>
            ) : (
              <div className="text-center py-16">
                <div className="w-16 h-16 mx-auto mb-4 rounded-full bg-gray-100 flex items-center justify-center">
                  <Calendar className="w-8 h-8 text-gray-400" />
                </div>
                <h3 className="text-lg font-medium text-gray-900 mb-2">No events found</h3>
                <p className="text-gray-500">
                  Try adjusting your filters or search terms
                </p>
              </div>
            )}
          </section>
        </div>
      </main>

      <Footer />
    </div>
  );
}
